#pragma once
#include "graph.hpp"
#include <vector>
#include <stack>
#include <algorithm>
#include <set>
#include <functional>

// Simple DFS-based cycle enumeration
// Returns a vector of cycles, each cycle is a vector of vertex indices
inline std::vector<std::vector<int>> enumerate_cycles(const Graph& g) {
    std::vector<std::vector<int>> cycles;
    std::vector<int> stack;
    std::vector<int> parent(g.n, -1);
    std::vector<bool> on_stack(g.n, false);

    std::function<void(int, int)> dfs = [&](int v, int pe) {
        stack.push_back(v);
        on_stack[v] = true;

        for (auto e : g.adj[v]) {
            int u = e.to;
            if (e.id == pe) continue;  // skip edge to parent
            if (!on_stack[u] && parent[u] == -1) {
                parent[u] = v;
                dfs(u, e.id);
            }
            else if (on_stack[u]) {
                std::vector<int> cycle;
                auto it = std::find(stack.rbegin(), stack.rend(), u);
                for (auto rit = stack.rbegin(); rit != it + 1; ++rit)
                    cycle.push_back(*rit);
                cycle.push_back(u);
                cycles.push_back(cycle);
            }
        }

        stack.pop_back();
        on_stack[v] = false;
        };

    for (int i = 0; i < g.n; ++i)
        if (parent[i] == -1) {
            parent[i] = -2;  // root marker
            dfs(i, -1);
        }

    // And remove duplicates that are same if rotated correctly
    std::vector<std::vector<int>> unique_cycles;
    std::set<std::vector<int>> seen;
    for (auto& c : cycles) {
        std::vector<int> sorted_cycle = c;
        std::rotate(sorted_cycle.begin(),
            std::min_element(sorted_cycle.begin(), sorted_cycle.end()),
            sorted_cycle.end());
        if (!seen.count(sorted_cycle)) {
            seen.insert(sorted_cycle);
            unique_cycles.push_back(c);
        }
    }

    return unique_cycles;
}